/* loop sampler 
---------------------------------
	_key commands_: 
	file: S=saveAs L=load
	perform: 1..0,Q..P=mute_unmute_loops K=kick_onOff / m=mode
	interface: D=up C=down ARROWS=move_cursor / f=up_more v=down_more 
	player: -=decTempo ==incTempo / space=play/pause w=reset e=rewind r=fast_forward 
	editor: g=place_loop b=remove_loop [=scroll_left ]=scroll_right

	_display_:
	main: tempo samples song_position
	track: # offset volume on/off loop_length
*/

#include <stdlib.h>
#include <windows.h>
#include <winbase.h>
#include <iostream>
#include <fstream>
#include "SDL/SDL.h"
#include "SDL/SDL_audio.h"
#include "SDL/SDL_TTF.h"
using namespace std;

//#define MAIN_SAMPLE_FILENAME "lion.wav"
#define MAIN_SAMPLE_FILENAME "sample6.wav"

#define RAND_SEED 65015
#define OFFSET_MULT 768

#define SAMPLE_RATE 44100

// these are for the audio engine startup
unsigned int sampleFrequency = 0;
unsigned int audioBufferSize = 0;
unsigned int outputAudioBufferSize = 0;

// amount to shift sample start point with keypress
#define SHIFT_AMOUNT SAMPLE_RATE/32

#define NUM_TRACKS 42
#define NUM_TRACKS_TO_MIX 41
#define NUM_SAMPLE_LOOPS 40
#define DRUM_CHANNEL 40
#define SAMPLE_FILES 4

#define FADE_SEG_SIZE (loopLength/NUM_TRACKS_TO_MIX)

#define MAX_VOLUME 128

char currentFile[256];
char sampleFile[256];

TTF_Font* font;

char keyKey[][22]={ "1","2","3","4","5","6","7","8","9","0","Q","W","E","R","T","Y","U","I","O","P"," "," "," ",
					" "," "," "," "," "," "," "," "," "," "," "," "," "," "," "," "," "," "," "," "," " };

#define SEQ_CURSOR 6
#define SEQ_DISP_COLUMN 51
int cursorLeft[]= {5, 9,16+ 9,16+24,16+30,16+35};
int cursorRight[]={7,23,16+22,16+28,16+33,16+39};

#define TRACK_DISPLAY_LINE 6

// TRACK_TYPE____________
struct trackType
{
	int sample;
	int volume;
	int volDir;
	bool enabled;
	Uint32 offset;
	int sampleDiv;
	char sequence[1024];
};
//^^^^^^^^^^^^^^^^^^^^^^

// SONG_TYPE____________
struct songType
{
	char file[20][20];
	int tempo;
	trackType track[NUM_TRACKS];
	char trackName[NUM_TRACKS][20];
}song;
//^^^^^^^^^^^^^^^^^^^^^^

// LOOP_TYPE____________
struct loopType
{
	Uint8* data;
	Uint32 dlen;
} loop[NUM_TRACKS];
//^^^^^^^^^^^^^^^^^^^^^^

SDL_Surface *screen;
bool seqMode=false;
bool playing=false;

#define MAX_LOOP_LENGTH (SAMPLE_RATE*32)

// loop/audio globals
Uint8 mainLoop[MAX_LOOP_LENGTH];
Uint32 loopPos = 0;
Uint32 measure = 0;
Uint32 totalMeasure = 0;
Uint32 seqDispWindow = 0;
bool redraw = false;

Uint32 callBacks = 0;

int currentSample = 0;

// primary sample sources - loaded at startup 
//Uint8* mainSample = 0;
Uint8* drumSample = 0;
Uint32 drumSampleLen = 0;

Uint8* mainSample[SAMPLE_FILES]={0,0,0,0};
Uint32 mainSampleLen[SAMPLE_FILES];

// this determines the tempo (range 8-32 / 240BPM - 60BPM)
Uint32 loopLength=SAMPLE_RATE*16;
// 120B/M = C*16
// 60B/M = C*16*2
// temp = 120/bpm*SAMPLE_RATE*16

#define DRUM_LENGTH (SAMPLE_RATE)


/* ========================================================================
   remix all loops 
=========================================================================== */
void remixLoops()
{
	Uint32 copyLength;
	Uint32 i, j;

	// clear main loop
	for( i=0; i<(loopLength/16*4); i++ )
		mainLoop[i] = 0;

	// mix all loops into main loop
	for( i=0; i<NUM_TRACKS; i++ )
	{
		if( (loop[i].data) && (song.track[i].sequence[totalMeasure]) )
		{
			if( loop[i].dlen > (loopLength/16*4) )
				copyLength = (loopLength/16*4);
			else
				copyLength = loop[i].dlen;

			SDL_MixAudio(mainLoop, &loop[i].data[(totalMeasure%4)*(loopLength/16*4)], (loopLength/16*4), song.track[i].volume);
//			SDL_MixAudio(&stream[streamPos], &loop[i].data[loopPos], amount, song.track[i].volume );

/*			for( j=0; j<NUM_TRACKS_TO_MIX; j++ )
			{
				SDL_MixAudio(&mainLoop[FADE_SEG_SIZE*j], &loop[i].data[FADE_SEG_SIZE*j], FADE_SEG_SIZE, song.track[i].volume);
			} */
		}

	}// for

//	reverb( (Uint16*) mainLoop, loopLength );
}

/* ========================================================================
 audio callback function 
=========================================================================== */
void callback(void *unused, Uint8 *stream, int len)
{
	int i=0;
	Uint32 amount;
	Uint32 tempLen;
	Uint32 streamPos;

	callBacks++;

	tempLen = len;
	streamPos = 0;

	if( !seqMode || playing )
	{
		while( tempLen > 0 )
		{
			amount = (loopLength/16*4)-loopPos;
			if ( amount > tempLen ) 
			{
				amount = tempLen;
			}
	
			for( i=0; i<NUM_TRACKS; i++ )
			{
				if(seqMode)
					memcpy(&stream[streamPos], &mainLoop[loopPos], amount);
				else
					if( (loop[i].data) && (song.track[i].enabled) )
						SDL_MixAudio(&stream[streamPos], &loop[i].data[loopPos+(measure%4)*(loopLength/16*4)], amount, song.track[i].volume );
			}
	
			loopPos += amount;
	
			tempLen = tempLen - amount;
			streamPos = streamPos + amount;
				
			// check for end of loop
			if( loopPos >= (loopLength/16*4) )
			{
				loopPos = 0;
				measure++;
				totalMeasure++;
				redraw=true;
				if( seqMode )
					remixLoops();
			}
	
		} // while
	}

} // mix function



/* ========================================================================
   Set up a Loop from a Sample
=========================================================================== */
void SetLoop( Uint8* sample, int loopNum, Uint32 offset, int volume, int div )
{
	int i;
	
	if( loopNum != DRUM_CHANNEL )
	{
		volume=80;
	
		div = song.track[loopNum].sampleDiv;
	
		offset = (offset % (mainSampleLen[song.track[loopNum].sample]-loopLength/div))/4*4;
	
		SDL_LockAudio();
		// free up the previous loop that was in this slot
		if ( loop[loopNum].data ) 
			free(loop[loopNum].data);
	
		// allocate buffer 
		loop[loopNum].data = (Uint8*) malloc( loopLength );
		SDL_UnlockAudio();
		
		if( sample )
		{
			for( i=0; i<div; i++ )
			{
				SDL_MixAudio( &loop[loopNum].data[(i*loopLength/div)/4*4], &sample[offset], loopLength/div, volume );
			}
		
			// save loop data into loop structure
		//	song.track[loopNum].volume = 128;
		//	song.track[loopNum].volDir = 1;
		
			song.track[loopNum].offset = offset;
	
		}
	}
}



/* ========================================================================
   Build kick drum
=========================================================================== */
void buildKick( int loopNum, int kickDrum, Uint32 pattern, bool rhythmKick )
{
	int i;
	Uint32 length;

	if( (loopLength/8) > DRUM_LENGTH )
		length = DRUM_LENGTH/2;
	else
		length = loopLength/8;


	SDL_LockAudio();
	// free up the previous loop that was in this slot
	if ( loop[loopNum].data ) 
		free(loop[loopNum].data);

	// allocate buffer 
	loop[loopNum].data = (Uint8*) malloc( loopLength );
	SDL_UnlockAudio();

	for( i=0; i<8; i++ )
	{
		SDL_MixAudio( &loop[loopNum].data[(loopLength/8*i)/4*4], drumSample, length, 128 );
	}

	// save loop data into loop structure
	song.track[loopNum].enabled = true;
	song.track[loopNum].volume = 128;
	song.track[loopNum].volDir = 1;
}


/* ========================================================================
   D R A W T E X T ()
=========================================================================== */
void drawText( SDL_Surface* screen, char* string, int color, int x, int y )
{
	Uint32 foreColors[]={0xFF0000,0x00FF00,0x0000FF,0xFFFF00,0x00FFFF,0xFF00FF,0x00FF00,0xFFFFFF, 
	                     0xFF0000,0x00FF00,0x0000FF,0xFFFF00,0x00FFFF,0xFF00FF,0x00FF00,0xFFFFFF,
						 0x666666,0x666666,0x000000,0x000000,0xaaaaaa,0xaaaaaa,0xffffff,0xffffff,
						 0xCCCCCC,0x00CCCC,0xCCCCCC,0x00CCCC,0xCCCCCC,0x00CCCC,0xCCCCCC,0x00CCCC,
						 0xFFFFFF,0x00FFFF,0xFFFFFF,0x00FFFF,0xFFFFFF,0x00FFFF,0xFFFFFF,0x00FFFF};

	Uint32 backColors[]={0x00FF00,0x0000FF,0xFF0000,0xFF00FF,0xFFFF00,0xFFFF00,0xFF00FF,0x000000, 
	                     0xFF00FF,0xFFFF00,0xFFFF00,0xFF00FF,0x000000,0x00FF00,0x0000FF,0xFF0000,
						 0xffffff,0x000000,0xaaaaaa,0xffffff,0x000000,0x666666,0x666666,0xaaaaaa,
						 0x000000,0x000000,0x440000,0x440000,0x004400,0x004400,0x000044,0x000044,
						 0x443333,0x443333,0x660000,0x660000,0x006600,0x006600,0x000066,0x000066};

	if( color>=40 )
		color = 0;

	x=x*8+10;
	y=y*15+10;

	int size = 14;

	SDL_Color foregroundColor = { 	foreColors[color]/0x10000%0x100, 
									foreColors[color]/0x100%0x100, 
									foreColors[color]/0x1%0x100 };

	SDL_Color backgroundColor = { 	backColors[color]/0x10000%0x100, 
									backColors[color]/0x100%0x100, 
									backColors[color]/0x1%0x100 };

	SDL_Surface* textSurface = TTF_RenderText_Shaded(font, string, foregroundColor, backgroundColor);
	SDL_Rect textLocation = { x, y, 0, 0 };
	SDL_BlitSurface(textSurface, NULL, screen, &textLocation);
	SDL_FreeSurface(textSurface);
}

/* ========================================================================
   INC LOOP
=========================================================================== */
void incLoop( int loopNum, int amount )
{
	SetLoop( mainSample[song.track[loopNum].sample], loopNum, song.track[loopNum].offset+SHIFT_AMOUNT*amount, 10, 4 );
}

/* ========================================================================
   DEC LOOP
=========================================================================== */
void decLoop( int loopNum, int amount )
{
	if( song.track[loopNum].offset > SHIFT_AMOUNT)
		SetLoop( mainSample[song.track[loopNum].sample], loopNum, song.track[loopNum].offset-SHIFT_AMOUNT*amount, 10, 4 );
}

/* ========================================================================
   incLength
=========================================================================== */
void incLength( int loopNum )
{
	switch( song.track[loopNum].sampleDiv )
	{
		case 1: song.track[loopNum].sampleDiv = 2; break;
		case 2: song.track[loopNum].sampleDiv = 4; break;
		case 4: song.track[loopNum].sampleDiv = 8; break;
		case 8: song.track[loopNum].sampleDiv = 16; break;
		case 16: song.track[loopNum].sampleDiv = 32; break;
		case 32: song.track[loopNum].sampleDiv = 1; break;
	}

	SetLoop( mainSample[song.track[loopNum].sample], loopNum, song.track[loopNum].offset, 10, 4 );
}


/* ========================================================================
   incLength
=========================================================================== */
void decLength( int loopNum )
{
	switch( song.track[loopNum].sampleDiv )
	{
		case 1: song.track[loopNum].sampleDiv = 32; break;
		case 2: song.track[loopNum].sampleDiv = 1; break;
		case 4: song.track[loopNum].sampleDiv = 2; break;
		case 8: song.track[loopNum].sampleDiv = 4; break;
		case 16: song.track[loopNum].sampleDiv = 8; break;
		case 32: song.track[loopNum].sampleDiv = 16; break;
	}

	SetLoop( mainSample[song.track[loopNum].sample], loopNum, song.track[loopNum].offset, 10, 4 );
}


/* ========================================================================
   MUTE TOGGLE
=========================================================================== */
void muteToggle( int loopNum )
{
	song.track[loopNum].enabled = !song.track[loopNum].enabled;
}


/* ========================================================================
   INPUT DATA
=========================================================================== */
int inputData( char* buffer, char* prompt )
{
	int returnValue = 1;
	SDL_Event event;
	bool runningInput=true;
	char key;
	int bufferPointer = 0;

	// clear buffer
	buffer[0] = 0;

#define INPUT_LINE 25
#define INPUT_OFFSET (strlen(prompt)+3)

	drawText( screen, prompt, 10, 0, 25 );
	drawText( screen, ">_______________<", 11, INPUT_OFFSET-1, 25 );
	SDL_Flip(screen);

	while( runningInput )
	{
		while (SDL_PollEvent(&event)) 
		{
			/* GLOBAL KEYS / EVENTS */
			switch (event.type) 
			{
				case SDL_KEYDOWN:
					
					key = event.key.keysym.sym;
					switch( key )
					{
						case SDLK_ESCAPE: 
							returnValue = 0;
							runningInput=false;
							break;
						case SDLK_RETURN: 
							runningInput=false;
							break;
						case SDLK_BACKSPACE:
							if( bufferPointer>0 )
							{
								bufferPointer--;
								buffer[bufferPointer]=0;
							}
							break;
						default:
							if( ((key>='a') && (key<='z')) || ((key>='A') && (key<='Z')) || ((key>=' ') && (key<='9')))
							{
								buffer[bufferPointer]=key;
								buffer[bufferPointer+1]=0;
								bufferPointer++;
							}
							break;
					}
			}
			SDL_Delay(5);
		}
		SDL_Delay(10);

		drawText( screen, ">_______________<", 11, INPUT_OFFSET-1, 25 );
		drawText( screen, buffer, 13, INPUT_OFFSET, 25 );
		SDL_Flip(screen);
			
	}

	drawText( screen, "                                ", 7, 0, 25 );
	drawText( screen, "                                ", 7, INPUT_OFFSET, 25 );
	SDL_Flip(screen);

	return( returnValue );
}


/* ========================================================================
   LOAD_SAMPLE
=========================================================================== */
void loadSample( int sampleNum )
{
	SDL_AudioSpec wave;
	char prompt[80];
	char buffer[80];

	sprintf( prompt, "LOAD SAMPLE #%d: Enter Filename", sampleNum+1 );

	if( inputData( buffer, prompt ) )
	{
		strcpy( song.file[sampleNum], buffer );
		strcat( song.file[sampleNum], ".wav" );
		if ( SDL_LoadWAV(song.file[sampleNum], &wave, &mainSample[sampleNum], &mainSampleLen[sampleNum]) == NULL ) 
			strcpy( song.file[sampleNum], "!error!" );
	}
}


/* ========================================================================
   CHANGE_TEMPO
=========================================================================== */
void changeTempo( int tempo )
{
	int loopNum;

	SDL_LockAudio();
	loopLength = (Uint32) ((120.0/tempo*SAMPLE_RATE*16))/4*4;

	// rebuild kick
	buildKick( DRUM_CHANNEL, 0, 0x11111111, true );

	// rebuild samples
	for(loopNum=0; loopNum<NUM_SAMPLE_LOOPS; loopNum++ )
	{
		if ( loop[loopNum].data ) 
			free(loop[loopNum].data);

		// allocate buffer 
		loop[loopNum].data = (Uint8*) malloc( loopLength );
	
		SetLoop( mainSample[song.track[loopNum].sample], loopNum, song.track[loopNum].offset, 10, 4 );
	}
	SDL_UnlockAudio();
}


/* ========================================================================
   M A I N ()
=========================================================================== */
int main( int argc, char* args[] ) 
{ 
	int cursorX = 0;
	int cursorY = 0;

	int tempo=120;
	song.tempo=120;

	Uint32 drawTimer=0;

	ofstream outfile;
	ifstream infile;
	char buffer[256];
	bool shift = false;
	int i;

	SDL_AudioSpec wave;
	char text[256];

	long sequence;

	if( SDL_Init(SDL_INIT_TIMER | SDL_INIT_AUDIO ) <0 ) 
	{
		cout << "Unable to init SDL: " << SDL_GetError() << endl;
		return 1;
	}

	TTF_Init();

	font = TTF_OpenFont("fixedsys.ttf", 14);
//	TTF_Font* font = TTF_OpenFont("amiga.ttf", 9);
	SDL_Color foregroundColor = { 255, 255, 255 };
	SDL_Color backgroundColor = { 0, 0, 0 };

	/* setup audio */
	SDL_AudioSpec *desired, *obtained;

	/* Allocate a desired SDL_AudioSpec */
	desired = (SDL_AudioSpec *) malloc(sizeof(SDL_AudioSpec));
	
	/* Allocate space for the obtained SDL_AudioSpec */
	obtained = (SDL_AudioSpec *) malloc(sizeof(SDL_AudioSpec));
	
	/* choose a samplerate and audio-format */
	desired->freq = 44100;
	desired->format = AUDIO_S16;
	
	/* Large audio buffers reduces risk of dropouts but increases response time.
	 *
	 * You should always check if you actually GOT the audiobuffer size you wanted,
	 * note that not hardware supports all buffer sizes (< 2048 bytes gives problems with some
	 * hardware). Older versions of SDL had a bug that caused many configuration to use a 
	 * buffersize of 11025 bytes, if your sdl.dll is approx. 1 Mb in stead of 220 Kb, download
	 * v1.2.8 of SDL or better...)
	 */
	desired->samples = 8192; 
	//desired->samples = 4192; 
	
	/* Our callback function */
	desired->callback=callback;
	desired->userdata=NULL;

	desired->channels = 2;

	/* Open the audio device and start playing sound! */
	if ( SDL_OpenAudio(desired, obtained) < 0 ) 
	{
		fprintf(stderr, "AudioMixer, Unable to open audio: %s\n", SDL_GetError());
		exit(1);
	}

	audioBufferSize = obtained->samples;
	sampleFrequency = obtained->freq;

	/* if the format is 16 bit, two bytes are written for every sample */
	if (obtained->format==AUDIO_U16 || obtained->format==AUDIO_S16) {
		outputAudioBufferSize = 2*audioBufferSize;
	} else {
		outputAudioBufferSize = audioBufferSize;
	}	

	// initialize sample div and volume of each track
	for( i=0; i<NUM_TRACKS; i++ )
	{
		song.track[i].sampleDiv = 4;
		song.track[i].volume = 128;
		song.track[i].sample = 0;
	}

	// Display Windows	
	screen = SDL_SetVideoMode(1000,740, 16, SDL_SWSURFACE);
	SDL_WM_SetCaption("L O O P   R E M I X E R",0);

	SDL_PauseAudio(0);

	SDL_EnableKeyRepeat(150, 30);

	seqMode = true;

	bool running = true;

	// Pre-load main sample
/*	strcpy( sampleFile, MAIN_SAMPLE_FILENAME );
	if ( SDL_LoadWAV(MAIN_SAMPLE_FILENAME, &wave, &mainSample, &mainSampleLen) == NULL ) {
		MessageBox (NULL, "Error", "Couldn't open main sample", MB_OK);
		printf( "Couldn't load drum_sample.wav: %s\n", SDL_GetError());
		return(-1);
	}
*/
	// Pre-load drum samples
	if ( SDL_LoadWAV("drum_sample.wav", &wave, &drumSample, &drumSampleLen) == NULL ) {
		MessageBox (NULL, "Error", "Couldn't open drum sample", MB_OK);
		printf( "Couldn't load drum_sample.wav: %s\n", SDL_GetError());
		return(-1);
	}

	changeTempo(tempo);
//	buildKick( DRUM_CHANNEL, 0, 0x11111111, true );

	// random seed
	srand(RAND_SEED);

	int oldMeasure = 0;
	measure = 5;	

	SDL_Event event;

	SDL_Flip(screen);

	Uint32 runningOffset=44100*40;

	for( i=0; i<NUM_SAMPLE_LOOPS; i++ )
		SetLoop( mainSample[0], i, rand()+rand()*32768, 10, 4 );

	while (running) 
	{
		if( measure != oldMeasure )
			oldMeasure = measure;

		drawTimer++;

		if( !(drawTimer%10) || redraw )
		{
			redraw = false;

			// display
			sprintf( text, "measure: %4.0d", totalMeasure );
			drawText( screen, text, 0, 0, 0 );

			if( playing )
				drawText( screen, "Seq: Play", 15, 15, 0 );
			else
				drawText( screen, "Seq: Stop", 15, 15, 0 );

			sprintf( text, "tempo: %d ", tempo );
			drawText( screen, text, 2, 27, 0 );
	
			if( seqMode )
				drawText( screen, "Mode: Sequencer  ", 15, 39, 1 );
			else
				drawText( screen, "Mode: Performance", 15, 39, 1 );
	
			if( song.track[DRUM_CHANNEL].enabled )
				drawText( screen, "Kick: ON ", 9, 0, 1 );
			else
				drawText( screen, "Kick: OFF", 9, 0, 1 );
	
			sprintf( text, "file: %s ", currentFile );
			drawText( screen, text, 13, 15, 1 ); 

			sprintf( text, "SAMP1: %s ", song.file[0] );
			drawText( screen, text, 6, 0, 2 );

			sprintf( text, "SAMP2: %s ", song.file[1] );
			drawText( screen, text, 6, 25, 2 );

			sprintf( text, "SAMP3: %s ", song.file[2] );
			drawText( screen, text, 13, 0, 3 );

			sprintf( text, "SAMP4: %s ", song.file[3] );
			drawText( screen, text, 13, 25, 3 );
			
	
			sprintf( text, "# |K|Sa#|Track Name     |Sample Offset | On  |Loop| Vol |%3.3d     %3.3d     %3.3d     %3.3d     %3.3d     %3.3d     %3.3d     %3.3d", 
						seqDispWindow, seqDispWindow+8, seqDispWindow+16, seqDispWindow+24, 
						seqDispWindow+32, seqDispWindow+40, seqDispWindow+48, seqDispWindow+56 );
			drawText( screen, text, 7, 0, TRACK_DISPLAY_LINE-1 );
	
			for( int displayLine=0; displayLine<NUM_TRACKS_TO_MIX; displayLine++ )
			{
				drawText( screen, "--|-|=-=|=-------------=|=------------=|=---=|=--=|=---=|+-------+-------+-------+-------+-------+-------+-------+-------",
									displayLine%2+24+song.track[displayLine].sample*2+song.track[displayLine].enabled*8,
									0,TRACK_DISPLAY_LINE+displayLine);
	
				int colorSet[]={	13, 3, 6, 5, 8, 3, 4, 6,12,14,
								16,17,18,19,20,21,22,23,16,17 };

				int colorOffset = 0;

				if( displayLine != DRUM_CHANNEL )
				{
					if( mainSample[song.track[displayLine].sample] )
						colorOffset = 0;	
					else
						colorOffset = 10;
	
					// draw a line where the sequence is currently playing
					if( seqMode )
						if( (totalMeasure>=seqDispWindow) && (totalMeasure<(seqDispWindow+64)) )
							drawText( screen, "-", 13, 16+41+totalMeasure-seqDispWindow, TRACK_DISPLAY_LINE+displayLine );
		
					// SAMPLE #
					sprintf( text, "%d", song.track[displayLine].sample+1 );
					drawText( screen, text, colorSet[colorOffset+song.track[displayLine].sample+1], 6, TRACK_DISPLAY_LINE+displayLine );
	
					// TRACK #
					sprintf( text, "%d", displayLine+1 ); 
					drawText( screen, text, colorSet[colorOffset+2], 0, TRACK_DISPLAY_LINE+displayLine );
		
					// KEY KEY
					drawText( screen, keyKey[displayLine], colorSet[colorOffset+3], 3, TRACK_DISPLAY_LINE+displayLine );

					// TRACK NAME
					drawText( screen, song.trackName[displayLine], colorSet[colorOffset+1+song.track[displayLine].sample*2], 10, TRACK_DISPLAY_LINE+displayLine );
		
					// SAMPLE OFFSET
					sprintf( text, "%d", song.track[displayLine].offset );
					drawText( screen, text, colorSet[colorOffset+4], 16+10, TRACK_DISPLAY_LINE+displayLine );
		
					// LOOP LENGTH
					sprintf( buffer, "%d", song.track[displayLine].sampleDiv );
					drawText( screen, buffer, colorSet[colorOffset+7], 16+31, TRACK_DISPLAY_LINE+displayLine );
		
					// VOLUME
					sprintf( buffer, "%d", song.track[displayLine].volume );
					drawText( screen, buffer, colorSet[colorOffset+8], 16+36, TRACK_DISPLAY_LINE+displayLine );
				}
				else
				{
					drawText( screen, "Drum Channel", 4, 7, TRACK_DISPLAY_LINE+displayLine );
				}

				if( !seqMode )
				{
					if( song.track[displayLine].enabled )
						drawText( screen, "ON ", 5, 16+25, TRACK_DISPLAY_LINE+displayLine );
					else
						drawText( screen, "OFF", 6, 16+25, TRACK_DISPLAY_LINE+displayLine );
				}
				else
				{
					if( song.track[displayLine].enabled )
						drawText( screen, "ON ", 18, 16+25, TRACK_DISPLAY_LINE+displayLine );
					else
						drawText( screen, "OFF", 20, 16+25, TRACK_DISPLAY_LINE+displayLine );
				}
			
				// show sequence marks
				for( int seqDisp=0; seqDisp<64; seqDisp++ )
					if( song.track[displayLine].sequence[seqDisp+seqDispWindow] )
						drawText( screen, "*", colorSet[colorOffset+9], 16+41+seqDisp, TRACK_DISPLAY_LINE+displayLine );
			}
	
			//display cursor
			if( cursorX<SEQ_CURSOR )
			{
				drawText( screen, "[", 15, cursorLeft[cursorX], cursorY+TRACK_DISPLAY_LINE );
				drawText( screen, "]", 15, cursorRight[cursorX], cursorY+TRACK_DISPLAY_LINE );
			}
			else
			{
				if( song.track[cursorY].sequence[cursorX-SEQ_CURSOR+seqDispWindow] )
					drawText( screen, "*", 15, cursorX+SEQ_DISP_COLUMN, cursorY+TRACK_DISPLAY_LINE );
				else
					drawText( screen, "-", 15, cursorX+SEQ_DISP_COLUMN, cursorY+TRACK_DISPLAY_LINE );
			}
	
			SDL_Flip(screen);
		}

		while (SDL_PollEvent(&event)) 
		{
			/* GLOBAL KEYS / EVENTS */
			switch (event.type) 
			{
				case SDL_KEYDOWN:
					redraw=true;
					switch (event.key.keysym.sym) 
					{
						// tempo +/-
						case '-': tempo--; changeTempo(tempo); song.tempo=tempo; break;
						case '=': tempo++; changeTempo(tempo); song.tempo=tempo; break;

						// shift (not implmented fully yet)
						case SDLK_LSHIFT:
						case SDLK_RSHIFT: shift = true; break;

						// cursor keys
						case SDLK_LEFT: 
							if( cursorX>0 ) cursorX--; break;
						case SDLK_RIGHT: 
							if( cursorX<69 ) cursorX++; break;
						case SDLK_UP: 
							if( cursorY>0 ) cursorY--; break;
						case SDLK_DOWN:
							if( cursorY<(NUM_TRACKS_TO_MIX-1)) cursorY++; break;

						case SDLK_HOME: cursorX=3; break;
						case SDLK_END: cursorX=38; break;


						case SDLK_ESCAPE: running = false; break;

						// toggle sequencer mode
						case 'm':
							seqMode = !seqMode;
							SDL_LockAudio();
							playing = false;
							loopPos = 0;	
							totalMeasure = 0;
							remixLoops();
							SDL_UnlockAudio();
							break;

						case 'g': //INC value
							switch( cursorX )
							{
								case 0: if( song.track[cursorY].sample<SAMPLE_FILES ) song.track[cursorY].sample++; 
									SetLoop( mainSample[song.track[cursorY].sample], cursorY, song.track[cursorY].offset, 10, 4 );
									break;
								case 1: if( inputData( buffer, "Enter Track Name" ) ) strcpy( song.trackName[cursorY], buffer ); break;
								case 2: incLoop(cursorY,1); break;
								case 3: muteToggle(cursorY); break;
								case 4: incLength(cursorY); break;
								case 5: if( song.track[cursorY].volume<255 ) song.track[cursorY].volume++; break;
								default:
									song.track[cursorY].sequence[cursorX-SEQ_CURSOR+seqDispWindow] = 1;
									break;
							}
							break;
						case 'b': //DEC value
							switch( cursorX )
							{
								case 0: if( song.track[cursorY].sample>0 ) song.track[cursorY].sample--;
									SetLoop( mainSample[song.track[cursorY].sample], cursorY, song.track[cursorY].offset, 10, 4 );
									break;
								case 1: if( inputData( buffer, "Enter Track Name" ) ) strcpy( song.trackName[cursorY], buffer ); break;
								case 2: decLoop(cursorY,1); break;
								case 3: muteToggle(cursorY); break;
								case 4: decLength(cursorY); break;
								case 5: if( song.track[cursorY].volume>0 ) song.track[cursorY].volume--; break;
								default:
									song.track[cursorY].sequence[cursorX-SEQ_CURSOR+seqDispWindow] = 0;
									break;
							}
							break;

						case 'd': //INC value
							switch( cursorX )
							{
								case 0: if( song.track[cursorY].sample<SAMPLE_FILES ) song.track[cursorY].sample++; 
									SetLoop( mainSample[song.track[cursorY].sample], cursorY, song.track[cursorY].offset, 10, 4 );
									break;
								case 1: if( inputData( buffer, "Enter Track Name" ) ) strcpy( song.trackName[cursorY], buffer ); break;
								case 2: incLoop(cursorY,4); break;
								case 3: muteToggle(cursorY); break;
								case 4: incLength(cursorY); break;
								case 5: if( song.track[cursorY].volume<255 ) song.track[cursorY].volume++; break;
								default:
									song.track[cursorY].sequence[cursorX-SEQ_CURSOR+seqDispWindow] = 1;
									break;
							}
							break;
						case 'c': //DEC value
							switch( cursorX )
							{
								case 0: if( song.track[cursorY].sample>0 ) song.track[cursorY].sample--;
									SetLoop( mainSample[song.track[cursorY].sample], cursorY, song.track[cursorY].offset, 10, 4 );
									break;
								case 1: if( inputData( buffer, "Enter Track Name" ) ) strcpy( song.trackName[cursorY], buffer ); break;
								case 2: decLoop(cursorY,4); break;
								case 3: muteToggle(cursorY); break;
								case 4: decLength(cursorY); break;
								case 5: if( song.track[cursorY].volume>0 ) song.track[cursorY].volume--; break;
								default:
									song.track[cursorY].sequence[cursorX-SEQ_CURSOR+seqDispWindow] = 0;
									break;
							}
							break;

						case 'f': //FAST INC value
							switch( cursorX )
							{
								case 0: if( song.track[cursorY].sample<SAMPLE_FILES ) song.track[cursorY].sample++;
									SetLoop( mainSample[song.track[cursorY].sample], cursorY, song.track[cursorY].offset, 10, 4 );
									break;
								case 1: if( inputData( buffer, "Enter Track Name" ) ) strcpy( song.trackName[cursorY], buffer ); break;
								case 2: incLoop(cursorY, 20); break;
								case 3: muteToggle(cursorY); break;
								case 4: incLength(cursorY); break;
								case 5: 
									if( song.track[cursorY].volume<245 ) 
										song.track[cursorY].volume+=10; 
									else
										song.track[cursorY].volume=255; 
									break;
								default:
									song.track[cursorY].sequence[cursorX-SEQ_CURSOR+seqDispWindow] = 1;
									break;
							}
							break;
						case 'v': //FAST DEC value
							switch( cursorX )
							{
								case 0: if( song.track[cursorY].sample>0 ) song.track[cursorY].sample--;
									SetLoop( mainSample[song.track[cursorY].sample], cursorY, song.track[cursorY].offset, 10, 4 );
									break;
								case 1: if( inputData( buffer, "Enter Track Name" ) ) strcpy( song.trackName[cursorY], buffer ); break;
								case 2: decLoop(cursorY,20); break;
								case 3: muteToggle(cursorY); break;
								case 4: decLength(cursorY); break;
								case 5: 
									if( song.track[cursorY].volume>10 ) 
										song.track[cursorY].volume-=10; 
									else
										song.track[cursorY].volume=0; 
								default:
									song.track[cursorY].sequence[cursorX-SEQ_CURSOR+seqDispWindow] = 0;
									break;
							}
							break;

						case SDLK_PAGEUP: 
							if( seqDispWindow>0 ) seqDispWindow-=8; break;

						case SDLK_PAGEDOWN: 
							if( seqDispWindow<(248-24) ) seqDispWindow+=8; break;

						// performance mode mute toggles for each track
						case '1': muteToggle(0); break;
						case '2': muteToggle(1); break;
						case '3': muteToggle(2); break;
						case '4': muteToggle(3); break;
						case '5': muteToggle(4); break;
						case '6': muteToggle(5); break;
						case '7': muteToggle(6); break;
						case '8': muteToggle(7); break;
						case '9': muteToggle(8); break;
						case '0': muteToggle(9); break;
						case 'q': muteToggle(10); break;
						case 'w': muteToggle(11); break;
						case 'e': muteToggle(12); break;
						case 'r': muteToggle(13); break;
						case 't': muteToggle(14); break;
						case 'y': muteToggle(15); break;
						case 'u': muteToggle(16); break;
						case 'i': muteToggle(17); break;
						case 'o': muteToggle(18); break;
						case 'p': muteToggle(19); break;

						case SDLK_F1: loadSample(0); break;
						case SDLK_F2: loadSample(1); break;
						case SDLK_F3: loadSample(2); break;
						case SDLK_F4: loadSample(3); break;

						case SDLK_F12: SetLoop( mainSample[song.track[cursorY].sample], cursorY, rand()+rand()*32768, 10, 4 ); break;

						// place current interactive mode data onto sequence
						case SDLK_TAB: 
							if( cursorX>4 )
								for( i=0; i<NUM_TRACKS_TO_MIX; i++ )
									song.track[i].sequence[cursorX-SEQ_CURSOR+seqDispWindow] = song.track[i].enabled;
							break;

						case SDLK_RETURN: 
							if( cursorX>4 )
								for( i=0; i<NUM_TRACKS_TO_MIX; i++ )
									song.track[i].enabled = song.track[i].sequence[cursorX-SEQ_CURSOR+seqDispWindow];
							seqMode = false;
							SDL_LockAudio();
							playing = false;
							loopPos = 0;	
							totalMeasure = 0;
							remixLoops();
							SDL_UnlockAudio();
							break;

										

						// kick drum on/off
						case 'k': song.track[DRUM_CHANNEL].enabled = !song.track[DRUM_CHANNEL].enabled; break;

						// save song to disk
						case 's': 
							if( inputData( buffer, "SAVE: Enter Filename" ) )
							{
								strcat(buffer,".lll");
								strcpy(currentFile, buffer);
								outfile.open( buffer, ios::binary );
								outfile.write( (char*) &song, sizeof(song) );
								outfile.close();
							}
							break;
	
						// load song from disk
						case 'l': 
							if( inputData( buffer, "LOAD: Enter Filename" ) ) 
							{
								strcat(buffer,".lll");
								strcpy(currentFile, buffer);
								infile.open( buffer, ios::binary );
								infile.read( (char*) &song, sizeof(song) );
								infile.close();
								for( i=0; i<4; i++ )
									if ( SDL_LoadWAV(song.file[i], &wave, &mainSample[i], &mainSampleLen[i]) == NULL ) 
										strcpy( song.file[i], "" );
								changeTempo(song.tempo);
								tempo=song.tempo;
							}
							break;

						// toggle start/stop sequencer
						case SDLK_SPACE:
							if( seqMode )
							{
								if( playing )
								{
									SDL_LockAudio();
									playing = false;
									loopPos = 0;	
									totalMeasure = 0;
									remixLoops();
									SDL_UnlockAudio();
								}
								else
								{
									loopPos = 0;	
									remixLoops();
									totalMeasure = 0;
									playing = true;
								}
							}
							else
							{
								seqMode = true;
								SDL_LockAudio();
								playing = false;
								loopPos = 0;	
								totalMeasure = 0;
								remixLoops();
								SDL_UnlockAudio();
							}
							break;
	
						case '`':
							if( !seqMode )
							{
								seqMode = true;
								SDL_LockAudio();
								loopPos = 0;	
								totalMeasure = 0;
								remixLoops();
								SDL_UnlockAudio();
							}		
							if( playing )
							{
								SDL_LockAudio();
								playing = false;
								loopPos = 0;	
								//totalMeasure = 0;
								remixLoops();
								SDL_UnlockAudio();
							}
							else
							{
								if( cursorX>5 )
									totalMeasure = cursorX-SEQ_CURSOR+seqDispWindow;
								else
									totalMeasure = 0;
								loopPos = 0;	
								remixLoops();
								playing = true;
							}
							break;
	
							

						default: 
							break;
					}
					break;

				case SDL_KEYUP:
					switch (event.key.keysym.sym) 
					{
						case SDLK_LSHIFT:
						case SDLK_RSHIFT: shift = false; break;
					}
					break;

				case SDL_QUIT:
					running = false;
					break;	  

			}

			SDL_Delay(1);
		}

		SDL_Delay(20);
	}

	TTF_Quit();
	SDL_Quit();

	return EXIT_SUCCESS;
}
